/* ==================================================================
File        : deferredFilterCommon.hlsl
Author      : Vassilis Poulopoulos
Date        : 15/07/2008 10:45:19
Format      : HLSL/Cg
Description : Common functions for deferred shadows & lights

================================================================= */

#if defined(CG_PS3)
	#define rm
	#define _register
#else
	#define _register register
	#define rm row_major
#endif

//------------------------------------------------------------------------------------------------
// Texture samplers
//------------------------------------------------------------------------------------------------

//@:-texformat g_StencilValuesTex RGBA8
//@:-texformat g_NormalsTex RGBA8
//@:-texformat g_LuminanceTex RGBA8
//@:-texformat g_AlbedoTex RGBA8

#if !defined(UBERSHADER)
sampler2D 	g_SceneDepthTex			: register(s0);	// Scene z buffer
sampler2D 	g_StencilValuesTex		: register(s1);	// Values that were stored in stencil
sampler2D 	g_NormalsTex			: register(s2);	// normal on the screen in eye space
sampler2D 	g_ShadowMapTex			: register(s4);
samplerCUBE g_ShadowMapCubeTex		: register(s4);
sampler2D 	g_LuminanceTex			: register(s5);
sampler2D 	g_AlbedoTex				: register(s6);
sampler2D 	g_SpotTex				: register(s7);
sampler2D 	g_NoiseTex				: register(s8);

#else
sampler2D	g_ShadowMapTex			: register(s15);
samplerCUBE	g_ShadowMapCubeTex		: register(s14);
#endif // UBERSHADER

//------------------------------------------------------------------------------------------------
// Engine parameters - mapped on to declarations in VertexShaderSemantics.fx
//------------------------------------------------------------------------------------------------

#include "VertexShaderSemantics.fx"

#define g_ScreenSize			vs_screenSize				// pixel width, height, 1/width, 1/height
#define g_ProjectionParams		vs_projection_params		// zNear, zFar, zFar-zNear, (zFar - zNear) / zFar
#define g_FrustumParams			vs_frustum_params			// left, width, bottom, height
#define g_ViewportScaleBias		vs_viewportScaleBias		// viewport scale xy, offset xy
#define g_InverseProjectionXY	vs_inverse_projection_xy	// conaints _00, _11, _20, _21 of the projection matrix

//------------------------------------------------------------------------------------------------
// Vertex formats
//------------------------------------------------------------------------------------------------

struct VS_IN
{
	float4 position	: POSITION;
#if defined(_WINPC)
	float2 tex		: TEXCOORD0;
#endif
};

struct VS_OUT
{
	float4 position				: POSITION;
	float4 coords				: TEXCOORD0;	// .xy Screen UV coords, .zw clip space coords [-1,1]
	float4 projectionParams		: TEXCOORD1;
};

struct VS_OUT_SM1
{
	float4 position				: POSITION;
	float2 coords0				: TEXCOORD0;
	float2 coords1				: TEXCOORD1;
};

//------------------------------------------------------------------------------------------------
// Helper functions
//------------------------------------------------------------------------------------------------

float sq(float a)
{ return  a*a; }

half sq(half a)
{ return  a*a; }

float2 sq(float2 a)
{ return  a*a; }

half2 sq(half2 a)
{ return  a*a; }

float sq4(float a)
{ return  sq(a)*sq(a); }

half sq4(half a)
{ return  sq(a)*sq(a); }

//------------------------------------------------------------------------------------------------
// Depth sampling functions
//------------------------------------------------------------------------------------------------

#if defined(CG_PS3)

	float color2depth(float4 depthColor)
	{
		float4 depthFactor = float4(256.0/16777215.0, 1.0/16777215.0, 0.0f, 256.0*256.0/16777215.0);
		return dot( round(depthColor*255.0), depthFactor );
	}

	float readSceneDepth(sampler2D tex, half2 uvTex)
	{
		return color2depth(tex2D(tex, uvTex).rgba);
	}

	float readShadowMap(sampler2D tex, half2 uvTex)
	{
		return color2depth(tex2D(tex, uvTex).rgba);
	}

	float readShadowMap(samplerCUBE tex, half3 uvTex)
	{
		return color2depth(texCUBE(tex, uvTex).rgba);
	}

#elif defined(_WINPC)

	// Used to sample from a buffer that returns a single float value (e.g. R32F, nVidia INTZ or ATI DF24)
	float readDepthFloat(sampler2D tex, float2 uvTex)
	{
		return tex2Dlod(tex, float4(uvTex,0,0)).r;
	}

	float readDepthFloat(samplerCUBE tex, float3 uvTex)
	{
		return texCUBE(tex, uvTex).r;
	}

	// Used when we have access to the DepthStencil buffer as an RGBA texture (i.e. nVidia RAWZ)
	float readDepthARG(sampler2D tex, float2 uvTex)
	{
		float4 depthColor = tex2Dlod(tex, float4(uvTex,0,0)).rgba;
		float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
		return dot(round(depthColor.arg*255.0), depthFactor);
	}

	float readDepthARG(samplerCUBE tex, float3 uvTex)
	{
		float4 depthColor = texCUBE(tex, uvTex).rgba;
		float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
		return dot(round(depthColor.arg*255.0), depthFactor);
	}

	// Used when we're manually rendered depth into an RGBA texture
	float readDepthRGB(sampler2D tex, float2 uvTex)
	{
		float3 depthColor = tex2D(tex, uvTex).rgb;
		float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
		return dot(round(depthColor*255.0), depthFactor);
	}

	float readDepthRGB(samplerCUBE tex, float3 uvTex)
	{
		float3 depthColor = texCUBE(tex, uvTex).rgb;
		float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
		return dot(round(depthColor*255.0), depthFactor);
	}


	// Supported combinations :
	// 0.Fallback					: FromColor,	FromColor,		Manual PCF
	// 1.nVidia 6xxx, 7xxx			: ARG,			ARG,			Hardware PCF
	// 2.nVidia 8xxx, ATI 4xxx+		: Float,		Float,			Hardware PCF
	// 3.ATI 1xxx+					: FromColor,	Float,			Fetch4 PCF
	// 4.ATI 2xxx+					: FromColor,	Float,			Hardware PCF

	#if defined(_SCENEZ_FROMCOLOR) && defined(_SHADOWZ_FROMCOLOR)

		float readSceneDepth(sampler2D tex, float2 uvTex)
		{
			return readDepthRGB(tex, uvTex);
		}

		float readShadowMap(sampler2D tex, float2 uvTex)
		{
			return readDepthRGB(tex, uvTex);
		}

		float readShadowMap(samplerCUBE tex, float3 uvTex)
		{
			// Cubemaps are always RGB
			return readDepthRGB(tex, uvTex);
		}

	#elif defined(_SCENEZ_ARG) && defined(_SHADOWZ_ARG)
		float readSceneDepth(sampler2D tex, float2 uvTex)
		{
			return readDepthARG(tex, uvTex);
		}

		float readShadowMap(sampler2D tex, float2 uvTex)
		{
			return readDepthARG(tex, uvTex);
		}

		float readShadowMap(samplerCUBE tex, float3 uvTex)
		{
			// Cubemaps are always RGB
			return readDepthRGB(tex, uvTex);
		}

	#elif defined(_SCENEZ_FLOAT) && defined(_SHADOWZ_FLOAT)

		float readSceneDepth(sampler2D tex, float2 uvTex)
		{
			return readDepthFloat(tex, uvTex);
		}

		float readShadowMap(sampler2D tex, float2 uvTex)
		{
			return readDepthFloat(tex, uvTex);
		}

		float readShadowMap(samplerCUBE tex, float3 uvTex)
		{
			// Cubemaps are always RGB
			return readDepthRGB(tex, uvTex);
		}

	#elif defined(_SCENEZ_FROMCOLOR) && defined(_SHADOWZ_FLOAT)

		float readSceneDepth(sampler2D tex, float2 uvTex)
		{
			return readDepthRGB(tex, uvTex);
		}

		float readShadowMap(sampler2D tex, float2 uvTex)
		{
			return readDepthFloat(tex, uvTex);
		}

		float readShadowMap(samplerCUBE tex, float3 uvTex)
		{
			// Cubemaps are always RGB
			return readDepthRGB(tex, uvTex);
		}

	#else

		// Error!
		float readSceneDepth(sampler2D tex, float2 uvTex)
		{
			return 0;
		}

		float readShadowMap(sampler2D tex, float2 uvTex)
		{
			return 0;
		}

		float readShadowMap(samplerCUBE tex, float3 uvTex)
		{
			return 0;
		}

	#endif

#else

	float readSceneDepth(sampler2D tex, half2 uvTex)
	{
		return tex2D(tex, uvTex).r;
	}

	float readShadowMap(sampler2D tex, half2 uvTex)
	{
		return tex2D(tex, uvTex).r;
	}

	float readShadowMap(samplerCUBE tex, half3 uvTex)
	{
		return texCUBE(tex, uvTex).r;
	}

#endif

#if !defined(UBERSHADER)

//------------------------------------------------------------------------------------------------
// BackProject
//------------------------------------------------------------------------------------------------
// Computes the fragment position in eye space.
// texCoord    coordinate of the fragment in texture/UV space ([0; 1] 1-[0; 1]).
// clipCoord   coordinate of the fragment in [-w/2, w/2, -h/2, h/2] space
float4 BackProject(float2 texCoord, float2 clipCoord, float4 projectionParams)
{
	// compute fragment position in eye space
	float zBufferValue = readSceneDepth(g_SceneDepthTex, texCoord);

	// backward projection
	float viewSpaceZ = projectionParams.x / (1.0 - zBufferValue * projectionParams.w);

	// thales theorem
	return float4(clipCoord * viewSpaceZ, viewSpaceZ, 1.0);
}

#endif // UBERSHADER

//------------------------------------------------------------------------------------------------
// Vertex shader to use for deferred shadow/light full screen quads
//------------------------------------------------------------------------------------------------
VS_OUT DeferredFilterQuadVS(VS_IN input)
{
	VS_OUT output;
	output.position = input.position;

#if defined(_WINPC)
	output.coords.xy = input.tex.xy;
	output.coords.zw = (input.tex.xy - 0.5) * half2(2.0, -2.0);
#else
	output.coords.xy = input.position.xy * half2(0.5, -0.5) + 0.5;
	output.coords.zw = input.position.xy;
#endif
	
	output.coords.xy = output.coords.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	// Undo projection to take us back to view space (bar the mulitply by W)
	output.coords.zw = output.coords.zw * g_InverseProjectionXY.xy + g_InverseProjectionXY.zw;

	output.projectionParams = g_ProjectionParams;
	
	return output;
}

//------------------------------------------------------------------------------------------------
// Vertex shader to use when rendering 3d geometry (e.g. light spheres/cones) for deferred shadows/lights
//------------------------------------------------------------------------------------------------
VS_OUT DeferredFilter3DVS(VS_IN input)
{
	float4 outPosition = mul(mul(input.position, world), viewProj);

	VS_OUT output;
	output.position = outPosition;

	// vertex coordinates in [0, 1] screen texture space, but before perspective-divide
	output.coords.xy = outPosition.xy * float2(0.5, -0.5) + 0.5*outPosition.w;
	output.coords.xy = output.coords.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy * outPosition.w;

	// Undo projection to take us back to view space (bar the mulitply by W)
	output.coords.zw = outPosition.xy * g_InverseProjectionXY.xy + g_InverseProjectionXY.zw * outPosition.w;

	// save projection params (reduce fragment shader patching on PS3)
	output.projectionParams = g_ProjectionParams;

	// save w so we can do perspective-divide in the fragment shader (cannot do earlier)
	// we HACK the y component (contains "far") because we won't use it in the fragment shader
	output.projectionParams.y = outPosition.w;

#if defined(_WINPC)
	output.position.x += -1.0 * g_ScreenSize.z * outPosition.w;
	output.position.y +=  1.0 * g_ScreenSize.w * outPosition.w;
#endif

	return output;
}

#if defined(_WINPC)

//------------------------------------------------------------------------------------------------
// Vertex shader used for rendering lights on SM1/SM2
//------------------------------------------------------------------------------------------------
VS_OUT_SM1 DeferredFilter2DSM1_vx(VS_IN input)
{
	VS_OUT_SM1 output;

	output.position	= input.position;
	output.coords0	= input.tex.xy;
	output.coords1	= input.tex.xy;

	return output;
}

//------------------------------------------------------------------------------------------------
// Pixel shader used for rendering lights on SM1/SM2
//------------------------------------------------------------------------------------------------
sampler2D 	g_SM1Lights_SceneTex	: register(s0);
sampler2D 	g_SM1Lights_LightingTex	: register(s1);

float4 DeferredFilter2DSM1_px(VS_OUT_SM1 input ) : COLOR
{
	float4 t0 = tex2D(g_SM1Lights_SceneTex,		input.coords0);
	float4 t1 = tex2D(g_SM1Lights_LightingTex,	input.coords1);

	return float4(t0.rgb * t1.rgb * 2.0, t0.a);
}

#endif // _WINPC
